/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Log.h"
#include "Subdivision.h"
#include "SliderSubdivision.h"
#include "WindowSystems/SliderInputEventDispatcher.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"

using namespace InputEventProcessing;
using namespace LayerManagerCalibration;

SliderInputEventDispatcher::SliderInputEventDispatcher()
    : InputEventDispatcher()
{
}

SliderInputEventDispatcher::~SliderInputEventDispatcher()
{
}

void SliderInputEventDispatcher::processTouchEvent(Subdivision* subdivision,
                                                   struct evdev_input_device* device,
                                                   coordinate& calibratedCoordinate,
                                                   struct TouchEvent& touchEvent)
{
    LOG_DEBUG("SliderInputEventDispatcher",
              "Process touch event for subdivision, name="
              << subdivision->getName()
              << ", coordinate: x="
              << calibratedCoordinate.x << ", y=" << calibratedCoordinate.y);

    // generate a coordinate relative to the origin of the subdivision
    coordinate relativeCoordinate = {
        calibratedCoordinate.x - subdivision->getTopLeftCoordinate().x,
        calibratedCoordinate.y - subdivision->getTopLeftCoordinate().y,
    };

    LOG_DEBUG("SliderInputEventEventDispatcher",
              "Relative coordinate: x="
              << relativeCoordinate.x << ", y=" << relativeCoordinate.y);

    // obtain input event
    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if (inputEvent == NULL)
    {
        LOG_WARNING("SliderInputEventDispatcher",
                    "Failed to process touch event");
    }
    else
    {
        SliderSubdivision *slider = dynamic_cast<SliderSubdivision*>(subdivision);

        if ((slider == NULL) || (getListener() == NULL))
        {
            LOG_WARNING("SliderInputEventDispatcher",
                        "Failed to process slider subdivision event");
        }
        else
        {
            uint sliderLength = (slider->getOrientation()
                                 == SliderSubdivision::HORIZONTAL)
                                ? subdivision->getWidth()
                                : subdivision->getHeight();

            if (sliderLength == 0)
            {
                LOG_WARNING("SliderInputEventDispatcher",
                            "Cannot process a slider of length 0");
            }
            else
            {
                uint sliderPosition = (slider->getOrientation()
                                       == SliderSubdivision::HORIZONTAL)
                                      ? relativeCoordinate.x
                                      : sliderLength - relativeCoordinate.y;

                float sliderScale = (slider->getMaximumValue()
                                     - slider->getMinimumValue())
                                    / (float)sliderLength;

                float value = (sliderScale * sliderPosition)
                               + slider->getMinimumValue();

                bool slotPopulated = false;
                bool slotStateValid = false;

                getSlotState(subdivision, device, touchEvent.slot,
                             slotPopulated, slotStateValid);

                LOG_DEBUG("SliderInputEventEventDispatcher",
                          "Slider position="
                          << sliderPosition
                          << ", scale="
                          << sliderScale
                          << ", slider minimum value="
                          << slider->getMinimumValue()
                          << ", slider maximum value="
                          << slider->getMaximumValue()
                          << ", touch event state="
                          << touchEvent.state);

                switch (touchEvent.state)
                {
                    case INPUT_STATE_PRESSED:
                        handleInputStatePressed(subdivision,
                                                device,
                                                slider,
                                                touchEvent,
                                                slotPopulated,
                                                slotStateValid,
                                                value);
                        break;

                    case INPUT_STATE_RELEASED:
                        handleInputStateReleased(subdivision,
                                                 device,
                                                 slider,
                                                 touchEvent,
                                                 slotPopulated,
                                                 slotStateValid);
                        break;

                    case INPUT_STATE_MOTION:
                        handleInputStateMotion(device,
                                               slider,
                                               touchEvent,
                                               slotPopulated,
                                               slotStateValid,
                                               value);
                        break;

                    default:
                        LOG_WARNING("SliderInputEventEventDispatcher",
                                    "Invalid touch event state="
                                    << touchEvent.state);
                        break;
                }
            }
        }
    }
}

void SliderInputEventDispatcher::processEnter(LayerManagerCalibration::Subdivision* subdivision,
                                              struct evdev_input_device* device,
                                              LayerManagerCalibration::coordinate& calibratedCoordinate,
                                              struct TouchEvent& touchEvent)
{
	(void) device;
	(void) calibratedCoordinate;
	(void) touchEvent;
    LOG_DEBUG("SliderInputEventDispatcher",
              "Process enter for subdivision, "
              "name=" << subdivision->getName());
}

void SliderInputEventDispatcher::processLeave(LayerManagerCalibration::Subdivision* subdivision,
                                              struct evdev_input_device* device,
                                              LayerManagerCalibration::coordinate& calibratedCoordinate,
                                              struct TouchEvent& touchEvent)
{
	(void) calibratedCoordinate;
    LOG_DEBUG("SliderInputEventDispatcher",
              "Process leave for subdivision, "
              "name="<< subdivision->getName());

    // obtain input event
    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if ((inputEvent == NULL) || (getListener() == NULL))
    {
        LOG_WARNING("SliderInputEventDispatcher",
                    "Failed to process touch event");
    }
    else
    {
        SliderSubdivision *slider = dynamic_cast<SliderSubdivision*>(subdivision);

        if (slider == NULL)
        {
            LOG_WARNING("SliderInputEventDispatcher",
                        "Failed to get slider subdivision");
        }
        else
        {
            bool slotPopulated = false;
            bool slotStateValid = false;

            getSlotState(subdivision, device, touchEvent.slot,
                         slotPopulated, slotStateValid);

            handleInputStateReleased(subdivision,
                                     device,
                                     slider,
                                     touchEvent,
                                     slotPopulated,
                                     slotStateValid);
            
        }
    }
}

void SliderInputEventDispatcher::getSlotState(LayerManagerCalibration::Subdivision* subdivision,
                                              struct evdev_input_device* device,
                                              uint slot,
                                              bool& slotPopulated,
                                              bool& slotStateValid)
{
    std::map<uint,bool>* slotMap;

    slotPopulated = false;
    slotStateValid = false;

    if (getState().getSubdivisionSlotMap(device,subdivision, &slotMap))
    {
        if (slotMap->size() != 0)
        {
            bool foundActiveSlot = false;
            uint activeSlot = 0;

            for (std::map<uint,bool>::iterator itr = slotMap->begin();
                                               itr != slotMap->end();
                                               itr++)
            {
                if (itr->second)
                {
                    foundActiveSlot = true;
                    activeSlot = itr->first;
                }

                if (itr->first == slot)
                {
                    slotPopulated = true;
                }
            }

            slotStateValid = (foundActiveSlot
                              && (activeSlot == slot));
        }
        else
        {
            slotStateValid = true;
        }

    }
    else
    {
        slotStateValid = true;
    }
}

void SliderInputEventDispatcher::handleInputStatePressed(LayerManagerCalibration::Subdivision* subdivision,
                                                         struct evdev_input_device* device,
                                                         LayerManagerCalibration::SliderSubdivision *slider,
                                                         struct TouchEvent& touchEvent,
                                                         bool& slotPopulated,
                                                         bool& slotStateValid,
                                                         float value)
{
    LOG_DEBUG("SliderInputEventEventDispatcher",
              "device name=" << device->deviceName << ", "
              "subdivision name="<< subdivision->getName() << ", "
              "slot no.="<< touchEvent.slot << ", "
              "slot populated=" << slotPopulated << ", "
              "value=" << value);

    if (!slotPopulated)
    {
        getState().setSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot,
                                             true);
        getState().setSubdivisionSlotValidity(device,
                                              subdivision,
                                              touchEvent.slot,
                                              slotStateValid);

        LOG_DEBUG("SliderInputEventEventDispatcher",
                  "Slot pressed and validity set, "
                  "device name=" << device->deviceName << ", "
                  "subdivision name="<< subdivision->getName() << ", "
                  "slot no.="<< touchEvent.slot << ", "
                  "slot state validity=" << slotStateValid);

        if (slotStateValid)
        {
            getListener()->generateButtonEvent(device,
                                               touchEvent.time,
                                               -1,
                                               -1,
                                               slider->getId(),
                                               WL_POINTER_BUTTON_STATE_PRESSED);
            getListener()->generateAxisEvent(device,
                                             touchEvent.time,
                                             slider->getId(),
                                             value);
        }
    }
}

void SliderInputEventDispatcher::handleInputStateReleased(LayerManagerCalibration::Subdivision* subdivision,
                                                          struct evdev_input_device* device,
                                                          LayerManagerCalibration::SliderSubdivision *slider,
                                                          struct TouchEvent& touchEvent,
                                                          bool& slotPopulated,
                                                          bool& slotStateValid)
{
    LOG_DEBUG("SliderInputEventEventDispatcher",
              "device name=" << device->deviceName << ", "
              "subdivision name="<< subdivision->getName() << ", "
              "slot no.="<< touchEvent.slot << ", "
              "slot populated=" << slotPopulated);

    if (slotPopulated)
    {
        getState().setSubdivisionSlotValidity(device,
                                              subdivision,
                                              touchEvent.slot,
                                              slotStateValid);
        getState().setSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot,
                                             false);

        LOG_DEBUG("SliderInputEventEventDispatcher",
                  "Slot pressed and validity set, "
                  "device name=" << device->deviceName << ", "
                  "subdivision name="<< subdivision->getName() << ", "
                  "slot no.="<< touchEvent.slot << ", "
                  "slot state validity=" << slotStateValid);

        if (slotStateValid)
        {
            getListener()->generateButtonEvent(device,
                                               touchEvent.time,
                                               -1,
                                               -1,
                                               slider->getId(),
                                               WL_POINTER_BUTTON_STATE_RELEASED);
        }
    }
}

void SliderInputEventDispatcher::handleInputStateMotion(struct evdev_input_device* device,
                                                        LayerManagerCalibration::SliderSubdivision *slider,
                                                        struct TouchEvent& touchEvent,
                                                        bool& slotPopulated,
                                                        bool& slotStateValid,
                                                        float value)
{
    LOG_DEBUG("SliderInputEventEventDispatcher",
              "Slot pressed and validity set, "
              "device name=" << device->deviceName << ", "
              "subdivision name="<< slider->getName() << ", "
              "slot no.="<< touchEvent.slot << ", "
              "slot populated=" << slotPopulated << ", "
              "slot state validity=" << slotStateValid << ", "
              "value=" << value);

    if (slotPopulated && slotStateValid)
    {
        getListener()->generateAxisEvent(device,
                                         touchEvent.time,
                                         slider->getId(),
                                         value);
    }
}
